<?php
// WordPress根目录PHP文件重命名工具
// 将文件保存为rename.php放在WordPress根目录运行

// 安全验证 - 必须直接访问
if (isset($_SERVER['SCRIPT_FILENAME']) && basename(__FILE__) === basename($_SERVER['SCRIPT_FILENAME'])) {
    processRename();
} else {
    exit('非法访问！');
}

function processRename() {
    // 开始输出HTML
    echo '<!DOCTYPE html>
    <html lang="zh-CN">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>WordPress PHP文件重命名工具</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 40px; line-height: 1.6; background: #f4f4f4; }
            .container { max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
            h1 { color: #333; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; }
            .warning { background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin: 20px 0; }
            .success { background: #d4edda; color: #155724; padding: 10px; border-radius: 5px; margin: 10px 0; }
            .error { background: #f8d7da; color: #721c24; padding: 10px; border-radius: 5px; margin: 10px 0; }
            button { background: #4CAF50; color: white; border: none; padding: 12px 24px; border-radius: 5px; cursor: pointer; font-size: 16px; }
            button:hover { background: #45a049; }
            button:disabled { background: #cccccc; cursor: not-allowed; }
            .file-list { background: #f8f9fa; padding: 15px; border-radius: 5px; margin: 15px 0; max-height: 300px; overflow-y: auto; }
            .file-item { padding: 5px 0; border-bottom: 1px solid #eee; }
            .original { color: #666; }
            .new { color: #28a745; font-weight: bold; }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>WordPress PHP文件重命名工具</h1>';
    
    // 处理表单提交
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm'])) {
        performRename();
    } else {
        showPreview();
    }
    
    echo '</div></body></html>';
}

function showPreview() {
    echo '<div class="warning">
            <strong>⚠️ 重要警告：</strong>
            <ul>
                <li>此操作会重命名WordPress根目录下的PHP文件（排除指定文件）</li>
                <li>操作前请<strong>务必备份</strong>您的网站文件和数据库</li>
                <li>重命名后可能会影响网站功能，请谨慎操作</li>
                <li>操作完成后请立即删除此脚本文件</li>
            </ul>
        </div>';
    
    // 获取文件列表预览
    $files = getFileList();
    
    if (empty($files)) {
        echo '<div class="error">没有找到需要重命名的PHP文件</div>';
        return;
    }
    
    echo '<h3>预览将重命名的文件 (' . count($files) . '个):</h3>
        <div class="file-list">';
    
    foreach ($files as $file) {
        $newName = generateNewName($file);
        echo '<div class="file-item">
                <span class="original">' . htmlspecialchars($file) . '</span> → 
                <span class="new">' . htmlspecialchars($newName) . '</span>
              </div>';
    }
    
    echo '</div>
        <form method="POST" onsubmit="return confirm(\'您确定要执行重命名操作吗？此操作不可撤销！\');">
            <button type="submit" name="confirm">确认执行重命名</button>
            <button type="button" onclick="window.location.reload()">刷新预览</button>
        </form>';
}

function performRename() {
    $files = getFileList();
    $successCount = 0;
    $errorCount = 0;
    $results = [];
    
    echo '<h3>执行重命名操作...</h3>';
    
    foreach ($files as $file) {
        $newName = generateNewName($file);
        
        if (rename($file, $newName)) {
            $successCount++;
            $results[] = '<div class="file-item success">
                            ✓ ' . htmlspecialchars($file) . ' → ' . htmlspecialchars($newName) . '
                          </div>';
        } else {
            $errorCount++;
            $results[] = '<div class="file-item error">
                            ✗ ' . htmlspecialchars($file) . ' 重命名失败
                          </div>';
        }
    }
    
    echo '<div class="' . ($successCount > 0 ? 'success' : 'error') . '">
            操作完成！成功: ' . $successCount . ' 个，失败: ' . $errorCount . ' 个
          </div>';
    
    echo '<div class="file-list">' . implode('', $results) . '</div>';
    
    echo '<div class="warning">
            <strong>操作完成提示：</strong>
            <ul>
                <li>请检查网站是否能正常访问</li>
                <li>如果网站无法访问，请恢复备份或手动重命名文件</li>
                <li><strong>请立即删除此脚本文件 (rename.php) 以确保安全</strong></li>
            </ul>
          </div>';
}

function getFileList() {
    // 排除的文件列表（WordPress核心文件）
    $excludedFiles = [
        'index.php',
        'wp-activate.php',
        'wp-blog-header.php',
        'wp-comments-post.php',
        'wp-config.php',        // 重要：必须排除配置文件
        'wp-cron.php',
        'wp-links-opml.php',
        'wp-load.php',
        'wp-login.php',
        'wp-mail.php',
        'wp-settings.php',
        'wp-signup.php',
        'wp-trackback.php',
        'xmlrpc.php',
        'wp-config-sample.php'
    ];
    
    $files = [];
    $dir = dirname(__FILE__);
    
    // 扫描当前目录（WordPress根目录）
    $allFiles = scandir($dir);
    
    foreach ($allFiles as $filename) {
        // 只处理.php文件
        if (pathinfo($filename, PATHINFO_EXTENSION) === 'php') {
            // 排除指定文件
            if (!in_array($filename, $excludedFiles, true)) {
                // 排除自身
                if ($filename !== basename(__FILE__)) {
                    $files[] = $filename;
                }
            }
        }
    }
    
    return $files;
}

function generateNewName($filename) {
    // 生成5-7位随机字母数字
    $length = rand(5, 7);
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $randomString = '';
    
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, strlen($characters) - 1)];
    }
    
    // 在扩展名前添加随机字符串
    $pathinfo = pathinfo($filename);
    return $pathinfo['filename'] . '_' . $randomString . '.' . $pathinfo['extension'];
}

// 兼容性处理 - 确保函数存在
if (!function_exists('random_int') && function_exists('rand')) {
    function random_int($min, $max) {
        return rand($min, $max);
    }
}
?>